<?php
/**
 * LayerPay - Crypto Payments for PrestaShop
 *
 * Accept ETH and USDC payments on Ethereum L2 networks
 * (Base, Optimism, Arbitrum) with 1% platform fee
 *
 * @author LayerPay
 * @copyright 2024 LayerPay
 * @license MIT
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

class LayerPay extends PaymentModule
{
    const LAYERPAY_NETWORKS_TEST = [
        'sepolia' => [
            'name' => 'Sepolia (Testnet)',
            'chain_id' => '0xaa36a7',
            'chain_id_dec' => 11155111,
            'contract' => '0x027811E894b6388C514f909d54921a701337f467',
            'usdc_address' => '0x7474e771f6f3d8123aa4cDD8d3593866651a14e6',
            'rpc_url' => 'https://ethereum-sepolia-rpc.publicnode.com',
            'explorer' => 'https://sepolia.etherscan.io',
            'currency' => 'ETH'
        ],
        'base_sepolia' => [
            'name' => 'Base Sepolia (Testnet)',
            'chain_id' => '0x14a34',
            'chain_id_dec' => 84532,
            'contract' => '0x34029850684D793012C277cB2ae4C4089213c0e9',
            'usdc_address' => '0x036CbD53842c5426634e7929541eC2318f3dCF7e',
            'rpc_url' => 'https://sepolia.base.org',
            'explorer' => 'https://sepolia.basescan.org',
            'currency' => 'ETH'
        ],
        'optimism_sepolia' => [
            'name' => 'OP Sepolia (Testnet)',
            'chain_id' => '0xaa37dc',
            'chain_id_dec' => 11155420,
            'contract' => '0xF0DCC0C62587804d9c49B075d24725A9a6eA2c6E',
            'usdc_address' => '0x0f411ff500f88BB528b800C7116c28d80f8BbD44',
            'rpc_url' => 'https://sepolia.optimism.io',
            'explorer' => 'https://sepolia-optimism.etherscan.io',
            'currency' => 'ETH'
        ],
        'arbitrum_sepolia' => [
            'name' => 'Arbitrum Sepolia (Testnet)',
            'chain_id' => '0x66eee',
            'chain_id_dec' => 421614,
            'contract' => '0xef0b1FA41570D81E0F1e834a85190FfC60F563AF',
            'usdc_address' => '0xd95480E52E671b87D6de3A3F05fbAb0E8526843F',
            'rpc_url' => 'https://sepolia-rollup.arbitrum.io/rpc',
            'explorer' => 'https://sepolia.arbiscan.io',
            'currency' => 'ETH'
        ]
    ];

    const LAYERPAY_NETWORKS_LIVE = [
        'ethereum' => [
            'name' => 'Ethereum',
            'chain_id' => '0x1',
            'chain_id_dec' => 1,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc_address' => '0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48',
            'rpc_url' => 'https://eth.llamarpc.com',
            'explorer' => 'https://etherscan.io',
            'currency' => 'ETH'
        ],
        'base' => [
            'name' => 'Base',
            'chain_id' => '0x2105',
            'chain_id_dec' => 8453,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc_address' => '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913',
            'rpc_url' => 'https://mainnet.base.org',
            'explorer' => 'https://basescan.org',
            'currency' => 'ETH'
        ],
        'optimism' => [
            'name' => 'Optimism',
            'chain_id' => '0xa',
            'chain_id_dec' => 10,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc_address' => '0x0b2C639c533813f4Aa9D7837CAf62653d097Ff85',
            'rpc_url' => 'https://mainnet.optimism.io',
            'explorer' => 'https://optimistic.etherscan.io',
            'currency' => 'ETH'
        ],
        'arbitrum' => [
            'name' => 'Arbitrum One',
            'chain_id' => '0xa4b1',
            'chain_id_dec' => 42161,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc_address' => '0xaf88d065e77c8cC2239327C5EDb3A432268e5831',
            'rpc_url' => 'https://arb1.arbitrum.io/rpc',
            'explorer' => 'https://arbiscan.io',
            'currency' => 'ETH'
        ]
    ];

    const ETH_PAYMENT_TOPIC = '0x4aa351061f13d3dff9e0f6cab4811de6a51a2f94e424b21ce31914f1e99c17bc';
    const TOKEN_PAYMENT_TOPIC = '0x0a7e11d6b5194b35bf3d4e463e2cb08dd9681b79fe6d4a1ff9725977a7da38d7';

    protected $html = '';
    protected $postErrors = [];

    public function __construct()
    {
        $this->name = 'layerpay';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'LayerPay';
        $this->need_instance = 1;
        $this->ps_versions_compliancy = ['min' => '1.7.0.0', 'max' => _PS_VERSION_];
        $this->bootstrap = true;
        $this->controllers = ['validation', 'convert'];

        parent::__construct();

        $this->displayName = $this->l('LayerPay - Crypto Payments');
        $this->description = $this->l('Accept ETH and USDC payments on Ethereum L2 networks (Base, Optimism, Arbitrum)');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall LayerPay?');

        if (!Configuration::get('LAYERPAY_MERCHANT_ADDRESS')) {
            $this->warning = $this->l('Please configure your merchant wallet address');
        }

        // Auto-register displayHeader hook if not already registered (for existing installations)
        if ($this->active && !$this->isRegisteredInHook('displayHeader')) {
            $this->registerHook('displayHeader');
        }
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        // Install database tables
        if (!$this->installDb()) {
            return false;
        }

        // Register hooks
        $hooks = [
            'paymentOptions',
            'paymentReturn',
            'displayAdminOrder',
            'displayOrderDetail',
            'actionFrontControllerSetMedia',
            'displayHeader',
            'moduleRoutes'
        ];

        foreach ($hooks as $hook) {
            if (!$this->registerHook($hook)) {
                return false;
            }
        }

        // Create order state for crypto payments
        $this->createOrderState();

        // Default configuration
        Configuration::updateValue('LAYERPAY_ENABLED', 1);
        Configuration::updateValue('LAYERPAY_NETWORK_MODE', 'test');
        Configuration::updateValue('LAYERPAY_PRICE_MARGIN', 2);
        Configuration::updateValue('LAYERPAY_PRICE_CACHE', 60);
        Configuration::updateValue('LAYERPAY_DEBUG', 0);

        return true;
    }

    protected function installDb()
    {
        $sql = file_get_contents(dirname(__FILE__) . '/sql/install.sql');
        if (!$sql) {
            return false;
        }

        $sql = str_replace('PREFIX_', _DB_PREFIX_, $sql);
        return Db::getInstance()->execute($sql);
    }

    protected function uninstallDb()
    {
        $sql = file_get_contents(dirname(__FILE__) . '/sql/uninstall.sql');
        if (!$sql) {
            return true;
        }

        $sql = str_replace('PREFIX_', _DB_PREFIX_, $sql);
        return Db::getInstance()->execute($sql);
    }

    public function uninstall()
    {
        // Remove database tables
        $this->uninstallDb();

        // Remove configuration
        $configs = [
            'LAYERPAY_ENABLED',
            'LAYERPAY_MERCHANT_ADDRESS',
            'LAYERPAY_NETWORK_MODE',
            'LAYERPAY_PRICE_MARGIN',
            'LAYERPAY_PRICE_CACHE',
            'LAYERPAY_DEBUG',
            'LAYERPAY_OS_CRYPTO_PENDING'
        ];

        foreach ($configs as $config) {
            Configuration::deleteByName($config);
        }

        return parent::uninstall();
    }

    protected function createOrderState()
    {
        $state = new OrderState();
        $state->name = [];
        foreach (Language::getLanguages() as $lang) {
            $state->name[$lang['id_lang']] = 'Awaiting crypto payment';
        }
        $state->color = '#FF9500';
        $state->hidden = false;
        $state->delivery = false;
        $state->logable = false;
        $state->invoice = false;
        $state->module_name = $this->name;
        $state->send_email = false;
        $state->paid = false;

        if ($state->add()) {
            Configuration::updateValue('LAYERPAY_OS_CRYPTO_PENDING', $state->id);
        }
    }

    public function hookModuleRoutes()
    {
        return [
            'module-layerpay-convert' => [
                'rule' => 'layerpay/convert',
                'keywords' => [],
                'controller' => 'convert',
                'params' => [
                    'fc' => 'module',
                    'module' => 'layerpay'
                ]
            ],
            'module-layerpay-validation' => [
                'rule' => 'layerpay/validation',
                'keywords' => [],
                'controller' => 'validation',
                'params' => [
                    'fc' => 'module',
                    'module' => 'layerpay'
                ]
            ]
        ];
    }

    public function getContent()
    {
        $this->html = '';

        if (Tools::isSubmit('submitLayerPayModule')) {
            $this->postValidation();
            if (!count($this->postErrors)) {
                $this->postProcess();
            } else {
                foreach ($this->postErrors as $err) {
                    $this->html .= $this->displayError($err);
                }
            }
        }

        $this->html .= $this->displayAdminInfo();
        $this->html .= $this->renderForm();

        return $this->html;
    }

    protected function displayAdminInfo()
    {
        $networkMode = Configuration::get('LAYERPAY_NETWORK_MODE');
        $isTestMode = $networkMode === 'test';

        $info = '<div class="panel">';
        $info .= '<h3><i class="icon-info-circle"></i> ' . $this->l('LayerPay Information') . '</h3>';
        $info .= '<p>' . $this->l('LayerPay enables crypto payments on Ethereum L2 networks with low gas fees.') . '</p>';
        $info .= '<ul>';
        $info .= '<li>' . $this->l('Supported networks: Base, Optimism, Arbitrum') . '</li>';
        $info .= '<li>' . $this->l('Supported currencies: ETH, USDC') . '</li>';
        $info .= '<li>' . $this->l('Platform fee: 1% (automatically deducted by smart contract)') . '</li>';
        $info .= '</ul>';

        if ($isTestMode) {
            $info .= '<div class="alert alert-warning">';
            $info .= '<strong>' . $this->l('TEST MODE ACTIVE') . '</strong> - ';
            $info .= $this->l('Using testnet networks. Switch to Live mode for production.');
            $info .= '</div>';
        }

        $info .= '</div>';

        return $info;
    }

    protected function renderForm()
    {
        $fields_form = [
            'form' => [
                'legend' => [
                    'title' => $this->l('LayerPay Settings'),
                    'icon' => 'icon-cogs'
                ],
                'input' => [
                    [
                        'type' => 'switch',
                        'label' => $this->l('Enable LayerPay'),
                        'name' => 'LAYERPAY_ENABLED',
                        'is_bool' => true,
                        'values' => [
                            ['id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')],
                            ['id' => 'active_off', 'value' => 0, 'label' => $this->l('No')]
                        ]
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Merchant Wallet Address'),
                        'name' => 'LAYERPAY_MERCHANT_ADDRESS',
                        'desc' => $this->l('Your Ethereum wallet address (0x...) to receive payments'),
                        'required' => true,
                        'class' => 'fixed-width-xxl'
                    ],
                    [
                        'type' => 'select',
                        'label' => $this->l('Network Mode'),
                        'name' => 'LAYERPAY_NETWORK_MODE',
                        'desc' => $this->l('Test mode uses testnet networks, Live mode uses mainnet'),
                        'options' => [
                            'query' => [
                                ['id' => 'test', 'name' => $this->l('Test (Testnets)')],
                                ['id' => 'live', 'name' => $this->l('Live (Mainnets)')]
                            ],
                            'id' => 'id',
                            'name' => 'name'
                        ]
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Price Margin (%)'),
                        'name' => 'LAYERPAY_PRICE_MARGIN',
                        'desc' => $this->l('Buffer percentage for price volatility (0-10%, default 2%)'),
                        'class' => 'fixed-width-sm',
                        'suffix' => '%'
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Price Cache Duration'),
                        'name' => 'LAYERPAY_PRICE_CACHE',
                        'desc' => $this->l('Cache ETH/USDC price for this many seconds (10-300)'),
                        'class' => 'fixed-width-sm',
                        'suffix' => 's'
                    ],
                    [
                        'type' => 'switch',
                        'label' => $this->l('Debug Mode'),
                        'name' => 'LAYERPAY_DEBUG',
                        'is_bool' => true,
                        'desc' => $this->l('Enable logging for troubleshooting'),
                        'values' => [
                            ['id' => 'debug_on', 'value' => 1, 'label' => $this->l('Yes')],
                            ['id' => 'debug_off', 'value' => 0, 'label' => $this->l('No')]
                        ]
                    ]
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right'
                ]
            ]
        ];

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = (int)Configuration::get('PS_LANG_DEFAULT');
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ?: 0;
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitLayerPayModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id
        ];

        return $helper->generateForm([$fields_form]);
    }

    protected function getConfigFormValues()
    {
        return [
            'LAYERPAY_ENABLED' => Configuration::get('LAYERPAY_ENABLED'),
            'LAYERPAY_MERCHANT_ADDRESS' => Configuration::get('LAYERPAY_MERCHANT_ADDRESS'),
            'LAYERPAY_NETWORK_MODE' => Configuration::get('LAYERPAY_NETWORK_MODE'),
            'LAYERPAY_PRICE_MARGIN' => Configuration::get('LAYERPAY_PRICE_MARGIN'),
            'LAYERPAY_PRICE_CACHE' => Configuration::get('LAYERPAY_PRICE_CACHE'),
            'LAYERPAY_DEBUG' => Configuration::get('LAYERPAY_DEBUG')
        ];
    }

    protected function postValidation()
    {
        $merchantAddress = Tools::getValue('LAYERPAY_MERCHANT_ADDRESS');

        if (!preg_match('/^0x[a-fA-F0-9]{40}$/', $merchantAddress)) {
            $this->postErrors[] = $this->l('Invalid merchant wallet address. Must be a valid Ethereum address (0x...)');
        }

        $margin = (float)Tools::getValue('LAYERPAY_PRICE_MARGIN');
        if ($margin < 0 || $margin > 10) {
            $this->postErrors[] = $this->l('Price margin must be between 0 and 10%');
        }

        $cache = (int)Tools::getValue('LAYERPAY_PRICE_CACHE');
        if ($cache < 10 || $cache > 300) {
            $this->postErrors[] = $this->l('Price cache duration must be between 10 and 300 seconds');
        }
    }

    protected function postProcess()
    {
        Configuration::updateValue('LAYERPAY_ENABLED', Tools::getValue('LAYERPAY_ENABLED'));
        Configuration::updateValue('LAYERPAY_MERCHANT_ADDRESS', Tools::getValue('LAYERPAY_MERCHANT_ADDRESS'));
        Configuration::updateValue('LAYERPAY_NETWORK_MODE', Tools::getValue('LAYERPAY_NETWORK_MODE'));
        Configuration::updateValue('LAYERPAY_PRICE_MARGIN', Tools::getValue('LAYERPAY_PRICE_MARGIN'));
        Configuration::updateValue('LAYERPAY_PRICE_CACHE', Tools::getValue('LAYERPAY_PRICE_CACHE'));
        Configuration::updateValue('LAYERPAY_DEBUG', Tools::getValue('LAYERPAY_DEBUG'));

        $this->html .= $this->displayConfirmation($this->l('Settings saved successfully'));
    }

    public function hookPaymentOptions($params)
    {
        if (!$this->active || !Configuration::get('LAYERPAY_ENABLED')) {
            return [];
        }

        if (!Configuration::get('LAYERPAY_MERCHANT_ADDRESS')) {
            return [];
        }

        $cart = $params['cart'];
        $currency = new Currency($cart->id_currency);

        // Only support EUR and USD for now
        if (!in_array($currency->iso_code, ['EUR', 'USD'])) {
            return [];
        }

        $paymentOption = new PaymentOption();
        $paymentOption
            ->setModuleName($this->name)
            ->setCallToActionText($this->l('Pay with Crypto (ETH/USDC)'))
            ->setAction($this->context->link->getModuleLink($this->name, 'validation', [], true))
            ->setAdditionalInformation($this->generatePaymentForm($cart, $currency))
            ->setBinary(true);

        return [$paymentOption];
    }

    protected function generatePaymentForm($cart, $currency)
    {
        $networkMode = Configuration::get('LAYERPAY_NETWORK_MODE');
        $isTestMode = $networkMode === 'test';
        $networks = $isTestMode ? self::LAYERPAY_NETWORKS_TEST : self::LAYERPAY_NETWORKS_LIVE;
        $merchantAddress = Configuration::get('LAYERPAY_MERCHANT_ADDRESS');
        $priceMargin = (float)Configuration::get('LAYERPAY_PRICE_MARGIN');

        $total = $cart->getOrderTotal(true, Cart::BOTH);

        $this->context->smarty->assign([
            'module_dir' => $this->_path,
            'networks' => $networks,
            'is_test_mode' => $isTestMode,
            'merchant_address' => $merchantAddress,
            'order_total' => $total,
            'currency_iso' => $currency->iso_code,
            'price_margin' => $priceMargin,
            'convert_url' => $this->context->link->getModuleLink($this->name, 'convert', [], true),
            'validation_url' => $this->context->link->getModuleLink($this->name, 'validation', [], true),
            'walletconnect_project_id' => 'bde3cb22b79eeda9c1dcfce0e9e4cdbd',
            'cart_id' => $cart->id
        ]);

        return $this->fetch('module:layerpay/views/templates/front/payment_form.tpl');
    }

    /**
     * Inject Web3Modal as ES module in head (loads once, avoids custom element re-registration)
     */
    public function hookDisplayHeader($params)
    {
        if ($this->context->controller->php_self !== 'order') {
            return '';
        }

        $projectId = 'bde3cb22b79eeda9c1dcfce0e9e4cdbd';
        $networks = $this->getNetworks();
        $chains = [];
        foreach ($networks as $key => $network) {
            $chains[] = [
                'chainId' => $network['chain_id_dec'],
                'name' => $network['name'],
                'currency' => $network['currency'],
                'explorerUrl' => $network['explorer'],
                'rpcUrl' => $network['rpc_url']
            ];
        }
        $chainsJson = json_encode($chains);
        $firstChainId = !empty($chains) ? $chains[0]['chainId'] : 11155111;
        $firstRpc = !empty($chains) ? $chains[0]['rpcUrl'] : 'https://ethereum-sepolia-rpc.publicnode.com';

        return "<script type=\"module\">
import { createWeb3Modal, defaultConfig } from 'https://esm.sh/@web3modal/ethers@5.1.11';

const projectId = '{$projectId}';
const chains = {$chainsJson};

const metadata = {
    name: 'LayerPay',
    description: 'Crypto Payment',
    url: window.location.origin,
    icons: []
};

const ethersConfig = defaultConfig({
    metadata,
    enableEIP6963: true,
    enableInjected: true,
    enableCoinbase: false,
    rpcUrl: '{$firstRpc}',
    defaultChainId: {$firstChainId}
});

const modal = createWeb3Modal({
    ethersConfig,
    chains,
    projectId,
    enableAnalytics: false,
    themeMode: 'light'
});

window.web3Modal = modal;
window.web3ModalReady = true;
window.dispatchEvent(new CustomEvent('web3ModalReady', { detail: { modal } }));
</script>";
    }

    public function hookActionFrontControllerSetMedia($params)
    {
        if ($this->context->controller->php_self === 'order') {
            // Module JS (WalletConnect is loaded dynamically inside JS)
            $this->context->controller->registerJavascript(
                'layerpay-checkout',
                'modules/' . $this->name . '/views/js/layerpay-checkout.js',
                ['position' => 'bottom', 'priority' => 100]
            );

            // Module CSS
            $this->context->controller->registerStylesheet(
                'layerpay-style',
                'modules/' . $this->name . '/views/css/layerpay.css',
                ['media' => 'all', 'priority' => 100]
            );
        }
    }

    public function hookPaymentReturn($params)
    {
        if (!$this->active) {
            return '';
        }

        $order = $params['order'];

        $txHash = '';
        $network = '';
        $cryptoAmount = '';
        $paymentType = '';

        // Get order payment info from custom table or order message
        $payments = $order->getOrderPayments();
        if (!empty($payments)) {
            $txHash = $payments[0]->transaction_id;
        }

        $this->context->smarty->assign([
            'shop_name' => $this->context->shop->name,
            'tx_hash' => $txHash,
            'order_reference' => $order->reference
        ]);

        return $this->fetch('module:layerpay/views/templates/hook/payment_return.tpl');
    }

    public function hookDisplayAdminOrder($params)
    {
        $order = new Order($params['id_order']);

        if ($order->module !== $this->name) {
            return '';
        }

        $payments = $order->getOrderPayments();
        $txHash = !empty($payments) ? $payments[0]->transaction_id : '';

        // Get stored crypto payment data
        $paymentData = $this->getOrderPaymentData($order->id);

        $this->context->smarty->assign([
            'tx_hash' => $txHash,
            'payment_data' => $paymentData,
            'networks' => array_merge(self::LAYERPAY_NETWORKS_TEST, self::LAYERPAY_NETWORKS_LIVE)
        ]);

        return $this->fetch('module:layerpay/views/templates/admin/order_info.tpl');
    }

    public function hookDisplayOrderDetail($params)
    {
        $order = $params['order'];

        if ($order->module !== $this->name) {
            return '';
        }

        $payments = $order->getOrderPayments();
        $txHash = !empty($payments) ? $payments[0]->transaction_id : '';

        $paymentData = $this->getOrderPaymentData($order->id);

        $this->context->smarty->assign([
            'tx_hash' => $txHash,
            'payment_data' => $paymentData
        ]);

        return $this->fetch('module:layerpay/views/templates/front/order_detail.tpl');
    }

    public function getNetworks()
    {
        $networkMode = Configuration::get('LAYERPAY_NETWORK_MODE');
        return $networkMode === 'test' ? self::LAYERPAY_NETWORKS_TEST : self::LAYERPAY_NETWORKS_LIVE;
    }

    public function getNetwork($key)
    {
        $networks = $this->getNetworks();
        return isset($networks[$key]) ? $networks[$key] : null;
    }

    public function saveOrderPaymentData($orderId, $data)
    {
        $json = json_encode($data);
        Db::getInstance()->insert('layerpay_payments', [
            'id_order' => (int)$orderId,
            'tx_hash' => pSQL($data['tx_hash']),
            'network' => pSQL($data['network']),
            'crypto_amount' => pSQL($data['crypto_amount']),
            'payment_type' => pSQL($data['payment_type']),
            'payer_address' => pSQL($data['payer_address']),
            'verified' => (int)$data['verified'],
            'data_json' => pSQL($json),
            'created_at' => date('Y-m-d H:i:s')
        ], false, true, Db::ON_DUPLICATE_KEY);
    }

    public function getOrderPaymentData($orderId)
    {
        $result = Db::getInstance()->getRow(
            'SELECT * FROM `' . _DB_PREFIX_ . 'layerpay_payments` WHERE `id_order` = ' . (int)$orderId
        );
        return $result;
    }

    public function verifyTransactionOnchain($txHash, $networkKey, $expectedAmount, $paymentType = 'eth')
    {
        $network = $this->getNetwork($networkKey);
        if (!$network) {
            return ['success' => false, 'error' => 'Invalid network'];
        }

        $merchantAddress = strtolower(Configuration::get('LAYERPAY_MERCHANT_ADDRESS'));

        // Get transaction receipt
        $receipt = $this->rpcCall($network['rpc_url'], 'eth_getTransactionReceipt', [$txHash]);

        if (!$receipt) {
            return ['success' => false, 'error' => 'Transaction not found'];
        }

        // Check transaction status
        $status = hexdec($receipt['status']);
        if ($status !== 1) {
            return ['success' => false, 'error' => 'Transaction failed on blockchain'];
        }

        // Verify contract address
        $txTo = strtolower($receipt['to']);
        $expectedContract = strtolower($network['contract']);

        if ($txTo !== $expectedContract) {
            return ['success' => false, 'error' => 'Transaction to wrong contract'];
        }

        // Parse event logs
        $paymentData = $this->parsePaymentLogs($receipt['logs'], $paymentType);

        if (!$paymentData) {
            return ['success' => false, 'error' => 'Payment event not found in transaction'];
        }

        // Verify merchant address
        if (strtolower($paymentData['merchant']) !== $merchantAddress) {
            return ['success' => false, 'error' => 'Payment to wrong merchant'];
        }

        return [
            'success' => true,
            'data' => [
                'payer' => $paymentData['payer'],
                'merchant' => $paymentData['merchant'],
                'amount' => $paymentData['amount'],
                'block_number' => hexdec($receipt['blockNumber'])
            ]
        ];
    }

    protected function parsePaymentLogs($logs, $paymentType)
    {
        $topic = $paymentType === 'usdc' ? self::TOKEN_PAYMENT_TOPIC : self::ETH_PAYMENT_TOPIC;

        foreach ($logs as $log) {
            if (isset($log['topics'][0]) && strtolower($log['topics'][0]) === strtolower($topic)) {
                $payer = '0x' . substr($log['topics'][1], 26);
                $merchant = '0x' . substr($log['topics'][2], 26);

                $data = $log['data'];
                $dataHex = substr($data, 2);
                $chunks = str_split($dataHex, 64);

                return [
                    'payer' => $payer,
                    'merchant' => $merchant,
                    'amount' => hexdec($chunks[1] ?? '0')
                ];
            }
        }

        return null;
    }

    protected function rpcCall($rpcUrl, $method, $params = [])
    {
        $data = json_encode([
            'jsonrpc' => '2.0',
            'method' => $method,
            'params' => $params,
            'id' => 1
        ]);

        $ch = curl_init($rpcUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            $this->log('RPC Error: ' . $error);
            return null;
        }

        $result = json_decode($response, true);

        if (isset($result['error'])) {
            $this->log('RPC Error: ' . json_encode($result['error']));
            return null;
        }

        return $result['result'] ?? null;
    }

    public function log($message)
    {
        if (Configuration::get('LAYERPAY_DEBUG')) {
            PrestaShopLogger::addLog('LayerPay: ' . $message, 1, null, 'LayerPay', null, true);
        }
    }
}
